local license =
[[----------------------------------------------------------------------
  **** BEGIN LICENSE BLOCK ****
	A simple flocking simulator written in LUA.

    Copyright (C) 2010
	Eric Fredericksen
	www.pttpsystems.com
	eric@pttpsystems.com

	This file is part of Flocker.

    Flocker is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

  **** END LICENSE BLOCK ****
------------------------------------------------------------------------]]


require( "iuplua" 	)
require( "cdlua" 	)
require( "iupluacd" )
require("iupluagl")
require("luagl")
require("luaglu")
--[[
in waiting
require( "iupluacontrols" )
]]

require("flockerutils")
require("flockerengine")
require("flockerbirds")
require("flockerdb")
require("flockerogl")


--[[ ============================= USER INTERFACE CODE GOES BELOW HERE ============================= ]]

-- global scope object for our code; let's keep things tidy
assert (flocker, "Flocker UI: Failed to find global variable flocker")

flocker.ui = {}

flocker.ui.transitionFrames 	= 2
flocker.ui.runningSimulation	= false
flocker.ui.refreshDisplay		= false
flocker.ui.autoRun				= false
flocker.ui.advantage			= 1.2 -- user speed advanage factor when driving around the world

function flocker.ui.RefreshDisplay()

	--[[ this is the one single place where updating light
		positions to world coords makes most sense
		This covers UI controls as well as resizing the window
		]]
	local wx, wy, wz = flocker.toroidX, flocker.toroidY, flocker.toroidZ

	for __, light in pairs(flocker.ui.lights) do
		light:UpdateWorldPosition(	wx, wy, wz )
	end

	-- update birds to be sure
	flocker.ui.lblBirdCount.title		= string.format("%d birds", flocker.birdCount)
	flocker.ui.lblSimulationTime.title	= string.format("%.2f s (%d steps)", flocker.totalT/1000, flocker.iterations )

	if flocker.ui.runningSimulation and 0 == flocker.birdCount then
		-- automatically toggle the start behavior and turn off the simulation
		flocker.ui.btnStart:action()
	end

	if flocker.ui.refreshDisplay then flocker.ui.transitionFrames = 2 end
	local transition = flocker.ui.transitionFrames > 0
	if( transition ) then flocker.ui.transitionFrames = flocker.ui.transitionFrames - 1 end

	local doRefresh = 	transition
						or flocker.ui.autoRun
						or flocker.ui.runningSimulation
						or flocker.trackCamera
						or flocker.trackCursor
						or flocker.ui.refreshDisplay

	if doRefresh then
		flocker.simulator.RefreshDisplay()
	end

	flocker.ui.refreshDisplay = false

end

-- this object defines the simulator API; a default object is included to tell you what you are missing
flocker.simulator =
	{
		  Iterate = function()
				flocker.assert(false, "flocker.simulator: missing function Iterate")
			end
		, RefreshDisplay = function()
				flocker.assert(false, "flocker.simulator: missing function RefreshDisplay")
			end
	}

-- We use a timer that comes with user interface library.
flocker.ui.refreshTimer = iup.timer{ time=flocker.deltaT, run="YES" }
flocker.ui.refreshTimer.action_cb = function()

	flocker.ogl.avatar:UpdatePosition()

	if flocker.ui.runningSimulation then
		flocker.simulator.Iterate()
	end

	-- because birds can die, we need to refresh the display
	flocker.ui.RefreshDisplay( )

end


-- used for handling certain problematic debugging situations :)
function flocker.assert( assertionResult, stringMessage, optionalDumper )

	if not assertionResult then
		if flocker.ui.runningSimulation then
			flocker.ui.btnStart.action() -- stop the simulation
		end
		print( stringMessage )
		if optionalDumper and ("function" == type(optionalDumper)) then
			optionalDumper()
		end
		assert(assertionResult, stringMessage )
	end

end

-- need to grab this later so assign to variable
flocker.ui.lblBirdCount = iup.label
	{
		title="0 birds"
		, minsize="120x"
		, padding="3x3"
		, alignment="ACENTER:ACENTER"
	}

-- need to grab this later so assign to variable
flocker.ui.btnClearBirds = iup.button
	{
		title="Clear Birds"
		, tip = "Remove all the birds from the simulation"
		, expand = "HORIZONTAL"
		, action = function()
			-- dump the existing birds
			flocker.ClearBirds()
			flocker.ogl.avatar:SetPosition(flocker.toroidX/2, flocker.toroidY/2, 1)

			flocker.ui.refreshDisplay = true
			return iup.DEFAULT
		end
	}


-- need to grab this later so assign to variable
flocker.ui.btnAddBird = iup.button
	{
		title="Add A Bird"
		, tip = "Add a new bird with a random position and direction"
		, expand="HORIZONTAL"
		, action = function()
			flocker.createNewBird()
			flocker.ui.refreshDisplay = true
		end
	}



flocker.ui.valuechanged_cb = function(self)
	self._label.title = string.format(self._fmtstring or "ERROR", self.value or -1);
	return iup.DEFAULT
end


flocker.ui.CreateHelpButton = function( url )

	return iup.button{ title="?", button_cb = function(self, but, pressed, x, y, status)
			if iup.BUTTON1 == but and 1==pressed and x>0 and y>0 then iup.Help(url) end
			end
			}

end

flocker.ui.makeSliderControlSet = function( params )

	local lbl = iup.label
		{
			  minsize 	= params.lbl.minsize	or "150x"
			, padding 	= params.lbl.padding	or "3x3"
			, alignment = params.lbl.alignment	or "ACENTER:ACENTER"
			, tip		= params.lbl.tip		or nil
			, onclick	= params.lbl.onclick	or nil
			, title 	= string.format( params.fmtstring, params.init )
		}
	local val = iup.val
		{
			  params.val.orientation or "HORIZONTAL" --first param is a string
			, min 				= params.val.min
			, max 				= params.val.max
			, step				= ((params.val.max-params.val.min)>1) and (1.0/(params.val.max-params.val.min)) or 0.01
			, value 			= params.init
			, showticks			= params.val.ticks or 10
			, _label 			= lbl
			, _fmtstring 		= params.fmtstring
			, valuechanged_cb 	= params.val.valuechanged_cb or flocker.ui.valuechanged_cb
		}
	return lbl, val
end


flocker.ui.lblZDepth, flocker.ui.valZDepthSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "World Z-Depth = %d px", init = flocker.toroidZ
		, lbl = { tip = "The depth of the world when in 3D-flocking mode." }
		, val =	{	  min = 100, max = 1000
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.toroidZ = tonumber(self.value)
						flocker.WrapBirds()

						flocker.ui.refreshDisplay = true
					end
				}
	}

flocker.ui.lblFOV, flocker.ui.valFOVSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Field of View = %d deg", init = flocker.ogl.camera.fov
		, lbl = { tip = "The breadth of view when in 3D-flocking mode." }
		, val =	{	  min = 5, max = 100
					, valuechanged_cb = function(self)
						self.value = math.floor(self.value+0.5)
						flocker.ui.valuechanged_cb(self)
						flocker.ogl.camera.fov = tonumber(self.value)
						flocker.ui.refreshDisplay = true
					end
				}
	}


flocker.ui.lblFogIntensity, flocker.ui.valFogIntensitySlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Fog Inensity = %.2f", init = flocker.ogl.fogIntensity
		, lbl = { }
		, val =	{	  min = 0, max = 1.0
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.ogl.fogIntensity = tonumber(self.value)
						flocker.ui.refreshDisplay = true
					end
				}
	}


flocker.ui.lblRepel, flocker.ui.valRepelSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Repulsion Radius = %.1f px", init = flocker.repulseR
		, lbl = { tip = "Move away from birds that\nare this close or closer." }
		, val =	{	  min = 5, max = 30
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.repulseR = tonumber(self.value)
						flocker.ui.refreshDisplay = true
					end
				}
	}


flocker.ui.lblDetect, flocker.ui.valDetectSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Attraction Radius = %.1f px", init = flocker.detectR
		, lbl = { tip = "Move toward birds that are\nat least this close." }
		, val =	{	  min = 31, max = 100
--					, tip = "Modify the attraction region. Birds will steer toward neighbors who get this close or closer but are outside of the repulsion region."
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.detectR = tonumber(self.value)
						flocker.ui.refreshDisplay = true
					end
				}
	}


flocker.ui.lblDeltaPX, flocker.ui.valDeltaPXSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Delta-px = %.1f px", init = flocker.deltaPX
		, lbl = { tip = "How far a bird moves during each\niteration of the simulation." }
		, val =	{ min = 0, max = 9
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.deltaPX = tonumber(self.value)
					end
				}
	}


flocker.ui.lblDeltaT, flocker.ui.valDeltaTSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Delta-t = %d ms", init = flocker.deltaT
		, lbl = { tip = "The time slice for each\nstep of the simulation." }
		, val =
		{	min = 10, max = 200
			, valuechanged_cb = function(self)
				self.value = math.floor(self.value+0.5)
				flocker.ui.valuechanged_cb(self)
				flocker.deltaT = tonumber(self.value)
				local orig=flocker.ui.refreshTimer.run
				flocker.ui.refreshTimer.run="NO"
				flocker.ui.refreshTimer.time = flocker.deltaT
				flocker.ui.refreshTimer.run=orig;
			end
		}
	}


flocker.ui.lblCaffeine, flocker.ui.valCaffeineSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Caffeine = %.1f mg", init = flocker.caffeine
		, lbl = { tip = "Amount of randomness in direction.\nThink of this as curiosity." }
		, val =	{ min = 0, max = 45
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.caffeine 	= tonumber(flocker.ui.valCaffeineSlide.value)
					end
				}
	}

flocker.ui.lblAgility, flocker.ui.valAgilitySlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Agility = %.1f ergs", init = flocker.agility
		, lbl = { tip = "How easily a bird turns. Higher\nnumbers mean tighter turns." }
		, val =	{ min = 0, max = 30
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.agility 	= tonumber(flocker.ui.valAgilitySlide.value)
					end
				}
	}

flocker.ui.lblAntiSociality, flocker.ui.valAntiSocialitySlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "AntiSociality = %.1f ergs", init = flocker.antisociality
		, lbl = { tip = "The strength of a bird's desire\nto maintain personal space." }
		, val =	{ min = 0, max = 10
					, valuechanged_cb = function(self)
						flocker.ui.valuechanged_cb(self)
						flocker.antisociality = tonumber(flocker.ui.valAntiSocialitySlide.value)
					end
				}
	}



flocker.ui.tglPartisanShip  = iup.toggle
	{
		title = "Birds are partisan"
		, tip = "Birds only flock with their own type."
	}
function flocker.ui.tglPartisanShip:action( v )
	flocker.partisan = (v==1)
end


flocker.ui.tglAggressive  = iup.toggle
	{
		title = "Birds are aggressive"
		, tip = "Adult birds will fight in some circumstances."
	}
function flocker.ui.tglAggressive:action( v )
	flocker.aggressive = (v==1)
end


flocker.ui.tglShowRepel  = iup.toggle
	{
		title = "Show repulsion region"
		, tip = "Show or hide the repulsion region\nas a circle around each bird."
	}
function flocker.ui.tglShowRepel:action( v )
	flocker.showRepel = (v==1)
	flocker.ui.refreshDisplay = true
end

flocker.ui.tglShowDetect = iup.toggle
	{
		title = "Show attraction region"
		, tip = "Show or hide the attraction region region\nas a circle around each bird."
	}
function flocker.ui.tglShowDetect:action( v )
	flocker.showDetect = (v==1)
	flocker.ui.refreshDisplay = true
end

flocker.ui.tglEnableFog  = iup.toggle
	{
		title = "Enable OpenGL Fog"
		, tip = "Show or hide the fog when using OpenGL."
	}
function flocker.ui.tglEnableFog:action( v )
	flocker.ogl.enableFog = (v==1)
	flocker.ui.refreshDisplay = true
end



flocker.ui.tglThreeDeeFlocking  = iup.toggle
	{
		title = "Enable flocking in 3D"
		, tip = "Switch between 2D and 3D flocking.\nDirection and position Z-components\nare set to zero when switching to 2D."
	}
function flocker.ui.tglThreeDeeFlocking:action( v )
	flocker.threeDeeFlocking = (v==1)
	if not flocker.threeDeeFlocking then
		flocker.Move3DBirdsTo2D()
	end
	flocker.ui.refreshDisplay = true
end

flocker.ui.tglReflectBirds  = iup.toggle
	{
		title = "Reflect birds at boundary"
		, tip = "Choose between reflection and teleportation\nof birds when they reach a boundary."
	}
function flocker.ui.tglReflectBirds:action( v )
	flocker.reflectBirds = (v==1)
	flocker.ui.refreshDisplay = true
end

flocker.ui.lblGap, flocker.ui.valGapSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "World-box gap = %d px", init = flocker.ogl.worldBoxGap
		, lbl = { tip = "The offset of world-box edges relative to a perfect cube." }
		, val =	{	  min = -100, max = 100
					, valuechanged_cb = function(self)
						self.value = math.floor(self.value+0.5)
						flocker.ui.valuechanged_cb(self)
						flocker.ogl.worldBoxGap = tonumber(self.value)
						flocker.ui.refreshDisplay = true
					end
				}
	}


-- auto fill in helper tables
flocker.ui.lstBgColorInit =
	{
		  dropdown="YES"
		, value=1
		, tip = "Choose the display area background color."
	}
for k, c in ipairs(flocker.colorMap) do
	flocker.ui.lstBgColorInit[k] = c.n
end

-- create the color selector control
flocker.ui.lstBgColor = 	iup.list( flocker.ui.lstBgColorInit )

-- so we can immediatly see the background color change
function flocker.ui.lstBgColor:valuechanged_cb()
	local colorID = tonumber(self.value)
	if( colorID ) then
		flocker.bgColor = tonumber(colorID)
	end
	flocker.ui.refreshDisplay = true
end


--[[=============== Create utility functions for color picker building ===================]]
function flocker.ui.SetColor( example, red, green, blue )
	local r, g, b = red, green, blue
	if "table" == type(red) then r, g, b = math.floor(r[1]*255), math.floor(r[2]*255), math.floor(r[3]*255) end
	example.bgcolor = string.format("%d %d %d", r, g, b)
	example.value 	= string.format("R=%d  G=%d  B=%d", r, g, b)
	local fgc = math.sqrt(r%2 + g^2 + b^2)
	fgc = (fgc > 127.5) and 0 or 255
	example.fgcolor = string.format("%d %d %d", fgc, fgc, fgc )
end

function flocker.ui.MyColorGetter(example, lt)

	local startX, startY = flocker.ui.dialog.CenterIt( 460, 230 )

	local red, green, blue = iup.GetColor(
		startX, startY, math.floor(lt[1]*255), math.floor(lt[2]*255), math.floor(lt[3]*255)
		)
	if not red then return nil end
	lt[1], lt[2], lt[3] = red/255, green/255, blue/255
	flocker.ui.SetColor( example, red, green, blue )
end


--[[============ create color picker for GLOBAL ambient light ======================]]
flocker.ui.txtGlobalAmbient = iup.text { expand="HORIZONTAL", alignment="ACENTER", readonly="YES"
		, tip = "Lights all objects within a scene equally,\nbrightening them without adding shading.\nThis is additive to the individual lights." }
flocker.ui.SetColor( flocker.ui.txtGlobalAmbient, flocker.ogl.lighting.globalAmbient )
flocker.ui.helpGlobalAmbient = flocker.ui.CreateHelpButton("http://en.wikipedia.org/wiki/Available_light")

flocker.ui.btnGlobalAmbient = iup.button
	{ title="Global Ambient", minsize="60x", expand="HORIZONTAL", tip = "Click to change color." }
function flocker.ui.btnGlobalAmbient:action( )
	flocker.ui.MyColorGetter(flocker.ui.txtGlobalAmbient, flocker.ogl.lighting.globalAmbient )
	flocker.ui.refreshDisplay = true
end


--[[ HELPER FUNCTIONS FOR UI ACCESS CONTROL ]]
function flocker.ui.SpotlightUIAccess( container )
	local enabled = container.light.enabled and container.light.spotEnabled
	container.spotlightFrame.active	= enabled and "YES" or "NO"
end

function flocker.ui.LightUIAccess( container )
	local enabled = container.light.enabled
	container.positionFrame.active		= enabled and "YES" or "NO"
	container.colorFrame.active			= enabled and "YES" or "NO"
	container.attenuationFrame.active	= enabled and "YES" or "NO"
	container.tglSpotEnable.active		= enabled and "YES" or "NO"
	flocker.ui.SpotlightUIAccess( container )
end


function flocker.ui.CreateLightController( firstLight )

	flocker.assert( firstLight and "table" == type(firstLight), "flocker.ui.CreateLightController: bad light prototype")

	-- syntactic sucrose
	local lighting = flocker.ogl.lighting

	--[[================ create light as requested =========================]]
	local container = { light = firstLight }


	--[[================ create a toggle for light enablement =========================]]
	container.tglLightEnable  = iup.toggle
	{
		title = "Enable Light", tip = "Turn this light on and off."
		, value = container.light.enabled and 1 or 0
		, action = function (self, v )
			container.light.enabled = (v==1)
			-- manage UI controls enable/disable
			flocker.ui.LightUIAccess( container )
			flocker.ui.refreshDisplay = true
		end
	}

	--[[======================== CREATE LIGHT POSITIONAL CONTROLS =============================]]

	container.lblX, container.valXSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "X-pos = %.2f", init = container.light.position[1]
		, lbl = { tip = "Relative position along the X-axis of the viewing volume." }
		, val = { min = 0.0, max = 1.0 }
	}
	function container.valXSlide:valuechanged_cb()
		flocker.ui.valuechanged_cb(		container.valXSlide	)
		container.light.position[1] = tonumber( container.valXSlide.value ) -- keep relative position
		flocker.ui.refreshDisplay = true
	end

	container.lblY, container.valYSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Y-pos = %.2f", init = container.light.position[2]
		, lbl = { tip = "Relative position along the Y-axis of the viewing volume." }
		, val = { min = 0.0, max = 1.0 }
	}
	function container.valYSlide:valuechanged_cb()
		flocker.ui.valuechanged_cb(		container.valYSlide	)
		container.light.position[2] = tonumber( container.valYSlide.value ) -- keep relative position
		flocker.ui.refreshDisplay = true
	end

	container.lblZ, container.valZSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Z-pos = %.2f", init = container.light.position[3]
		, lbl = { tip = "Relative position along the Z-axis of the viewing volume." }
		, val = { min = 0.0, max = 1.0 }
	}
	function container.valZSlide:valuechanged_cb()
		flocker.ui.valuechanged_cb(		container.valZSlide	)
		container.light.position[3] = tonumber( container.valZSlide.value ) -- keep relative position
		flocker.ui.refreshDisplay = true
	end

	local SetColor = flocker.ui.SetColor
	local MyColorGetter = flocker.ui.MyColorGetter

	--[[============ CREATE COLOR PICKES FOR AMBIENT, DIFFUSE, SPECULAR LIGHT COLORS =========]]

	--[[ WE USE TEXT OBJECTS FOR COLOR BECAUSE WE CANNOT SET THE BACKGROUND COLOR OF LABEL OBJECTS 	]]

	container.txtDiffuse = iup.text { expand="HORIZONTAL", alignment="ACENTER", readonly="YES"
		, tip="Originates from the light position and\nspreads outward equally in all directions." }
	SetColor( container.txtDiffuse, container.light.diffuse )

	container.btnDiffuse = iup.button
		{ title="Diffuse", minsize="60x", expand="HORIZONTAL", tip = "Click to change color." }
	container.helpDiffuse = flocker.ui.CreateHelpButton("http://en.wikipedia.org/wiki/Diffuse_reflection")

	function container.btnDiffuse:action( )
		MyColorGetter( container.txtDiffuse, container.light.diffuse )
		flocker.ui.refreshDisplay = true
	end

	container.txtAmbient = iup.text { expand="HORIZONTAL", alignment="ACENTER", readonly="YES"
		, tip = "Lights all objects within a scene equally,\nbrightening them without adding shading." }
	SetColor( container.txtAmbient, container.light.ambient )

	container.btnAmbient = iup.button
		{ title="Ambient", minsize="60x", expand="HORIZONTAL", tip = "Click to change color." }
	container.helpAmbient = flocker.ui.CreateHelpButton("http://en.wikipedia.org/wiki/Shading#Light_sources")

	function container.btnAmbient:action( )
		MyColorGetter( container.txtAmbient, container.light.ambient )
		flocker.ui.refreshDisplay = true
	end

	container.txtSpecular = iup.text { expand="HORIZONTAL", alignment="ACENTER", readonly="YES"
		, tip = "The mirror-like reflection of light from a surface." }
	SetColor( container.txtSpecular, container.light.specular )
	container.helpSpecular = flocker.ui.CreateHelpButton("http://en.wikipedia.org/wiki/Specular")

	container.btnSpecular = iup.button
		{ title="Specular", minsize="60x", expand="HORIZONTAL", tip = "Click to change color."}
	function container.btnSpecular:action( )
		MyColorGetter( container.txtSpecular, container.light.specular )
		flocker.ui.refreshDisplay = true
	end


	--[[============================ ATTENUATION CONFIGURATION =====================================]]

	--[[============== create popuup dialog to get the three attenuation parameters ================]]
	function container.SetAttenuationText()
		local att = container.light.attenuation
		local title = string.format( "C=%.4f  L=%.4f  Q=%.4f", att[1], att[2], att[3] )
		container.lblAttenuation.title = title
	end

	container.lblAttenuation = iup.label
	{
		  expand="HORIZONTAL", alignment="ACENTER"
		, tip = "Modify the L and/or Q components to reduce light\nintensity as distance from the light increases."
	}
	container.SetAttenuationText()
	container.helpAttenuation = flocker.ui.CreateHelpButton("http://en.wikipedia.org/wiki/Shading#Distance_falloff")

	container.btnAttenuation = iup.button
		{ title="Edit Attenuation", minsize="60x", expand="HORIZONTAL", tip = "Click to modify attenuation." }
	function container.btnAttenuation:action()

		local C, L, Q = 	container.light.attenuation[1], container.light.attenuation[2], container.light.attenuation[3]

		C,L,Q = iup.Scanf(
				[[
				Attenuation with distance from light.
				Constant Component:%25.4%f
				Linear Component:%25.4%f
				Quadratic Component:%25.4%f
				]]
				, C, L, Q )

		C,L,Q = math.abs(C), math.abs(L), math.abs(Q)

		if (C + L + Q) == 0 then
			iup.Message("Bad parameter set.", "At least one of C, L, or Q must be non-zero.")
			return
		end

		if C then
			container.light.attenuation[1], container.light.attenuation[2], container.light.attenuation[3] = C,L,Q
			container.SetAttenuationText()
		end
		flocker.ui.refreshDisplay = true
	end



	--[[============================ SPOTLIGHT CONFIGURATION ========================================]]

	--[[================ create a toggle for spotlight enablement ===================================]]
	container.tglSpotEnable  = iup.toggle
	{
		title = "Enable Spotlight", tip = "Choose between an omnidirectional or a spot light source."
		, value = container.light.spotEnabled and 1 or 0
		, action = function (self, v )
			container.light.spotEnabled = (v==1)
			-- mange UI enable/disable
			flocker.ui.SpotlightUIAccess( container )
			flocker.ui.refreshDisplay = true
		end
	}


	--[[================ create sliders for PHI, THETA: spotlight direction =========================]]

	local __
	__, container.light.spotTheta, container.light.spotPhi =
		CartesianToSpherical( container.light.spotDirection, nil, nil, true  )

	container.lblPhi, container.valPhiSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Phi = %d deg", init = container.light.spotPhi, lbl = { }, val = { min = -180, max = 180 }
	}
	container.lblPhi.tip = "Rotation about the Z-axis as derived\nfrom spherical coordinates: Rho, Phi, Theta."
	container.valPhiSlide.pagestep = 10/360
	function container.valPhiSlide:valuechanged_cb()
		container.valPhiSlide.value = math.floor( 0.5 + container.valPhiSlide.value)
		flocker.ui.valuechanged_cb(container.valPhiSlide)
		container.light.spotPhi = tonumber(container.valPhiSlide.value)

		container.light.spotDirection[1], container.light.spotDirection[2], container.light.spotDirection[3] =
			SphericalToCartesian(1,  container.light.spotTheta, container.light.spotPhi, true --[[degrees]] )

		flocker.ui.refreshDisplay = true
	end

	container.lblTheta, container.valThetaSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Theta = %d deg", init = container.light.spotTheta, lbl = { }, val = { min = -180, max = 180 }
	}
	container.lblTheta.tip = "Rotation about the Y-axis as derived\nfrom spherical coordinates: Rho, Phi, Theta"
	container.valThetaSlide.pagestep = 10/360
	function container.valThetaSlide:valuechanged_cb()
		container.valThetaSlide.value = math.floor( 0.5 + container.valThetaSlide.value)
		flocker.ui.valuechanged_cb(container.valThetaSlide)
		container.light.spotTheta = tonumber(container.valThetaSlide.value)

		container.light.spotDirection[1], container.light.spotDirection[2], container.light.spotDirection[3] =
			SphericalToCartesian(1,  container.light.spotTheta, container.light.spotPhi, true --[[degrees]] )

		flocker.ui.refreshDisplay = true
	end

	--[[================ create spinner for spot exponent =========================================]]
	container.lblExp, container.valExpSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Exponent = %d", init = container.light.spotExponent
		, lbl = { tip = "Determines how quickly light intensity falls\noff smoothly from the center of the spot." }, val = { min = 0.0, max = 80 }
	}
	function container.valExpSlide:valuechanged_cb()
		flocker.ui.valuechanged_cb(container.valExpSlide)
		container.light.spotExponent = tonumber(container.valExpSlide.value)
		flocker.ui.refreshDisplay = true
	end

	--[[================ create slider for spot cutoff =============================================]]
	container.lblCutoff, container.valCutoffSlide = flocker.ui.makeSliderControlSet
	{
		fmtstring = "Cutoff angle = %d deg", init = container.light.spotCutoff
		, lbl = { tip = "The angle from the center of the spot\nat which the light is cut off sharply." }, val = { min = 0.0, max = 90 }
	}
	function container.valCutoffSlide:valuechanged_cb()
		flocker.ui.valuechanged_cb(container.valCutoffSlide)
		container.light.spotCutoff = tonumber(container.valCutoffSlide.value)
		flocker.ui.refreshDisplay = true
	end


	--[[================================ UI FRAME LAYOUT ==========================================]]

	container.colorFrame = iup.frame
	{	title = "Colors"
		, iup.vbox
		{
			  iup.hbox { container.btnAmbient,	container.helpAmbient,	container.txtAmbient  }
			, iup.hbox { container.btnDiffuse,	container.helpDiffuse,	container.txtDiffuse}
			, iup.hbox { container.btnSpecular,	container.helpSpecular, container.txtSpecular	}
		}
	}

	container.attenuationFrame = iup.frame
	{	title = "Attenuation = 1/(C + L*D + Q*D*D)"
		, iup.vbox
		{
			iup.vbox { iup.hbox{container.btnAttenuation, container.helpAttenuation}, container.lblAttenuation }
		}
	}

	container.positionFrame = iup.frame
	{	title = "Position"
		, iup.vbox
		{	alignment="ACENTER"
			, iup.hbox { iup.hbox{ iup.fill{}, container.valXSlide, iup.frame{container.lblX}, iup.fill{} } }
			, iup.hbox { container.valYSlide, iup.frame{container.lblY} }
			, iup.hbox { container.valZSlide, iup.frame{container.lblZ} }
		}
	}

	container.spotlightFrame = iup.frame
	{	title = "Spotlight Configuration"
		, iup.vbox
		{
			  alignment="ACENTER"
			, expand = "YES"
			-- put theta first because it is more intuitive given the 3D coordinate system
			, iup.hbox { iup.fill{}, container.valThetaSlide, 	iup.frame{container.lblTheta}, iup.fill{} }
			, iup.hbox { container.valPhiSlide, 	iup.frame{container.lblPhi} }
			, iup.hbox { container.valExpSlide, 	iup.frame{container.lblExp} }
			, iup.hbox { container.valCutoffSlide, 	iup.frame{container.lblCutoff} }

		}
	}



	local lightFrame = iup.frame
	{
		  title= nil --"Light Controls"
		, expand="YES"
		, container = container -- so I can edit the light later
		, iup.vbox
		{	expand="YES", alignment="ACENTER"
			, container.tglLightEnable
			, container.colorFrame
			, container.attenuationFrame
			, container.positionFrame
			, iup.hbox{ container.tglSpotEnable }
			, container.spotlightFrame
		}

	}
	function lightFrame.UpdateLightControls( theLight )

		-- update the reference to our light
		lightFrame.container.light = theLight

		local light = lightFrame.container.light

		-- update global ambient setting
		SetColor( flocker.ui.txtGlobalAmbient, flocker.ogl.lighting.globalAmbient )

		-- update enabled status
		container.tglLightEnable.value = light.enabled and 1 or 0

		-- update light position
		container.valXSlide.value = light.position[1]
		container.valXSlide.valuechanged_cb()

		container.valYSlide.value = light.position[2]
		container.valYSlide.valuechanged_cb()

		container.valZSlide.value = light.position[3]
		container.valZSlide.valuechanged_cb()

		-- update attenuation
		container.SetAttenuationText()

		SetColor( container.txtAmbient,		light.ambient )
		SetColor( container.txtDiffuse,		light.diffuse )
		SetColor( container.txtSpecular,	light.specular )

		-- update spotlight enabled status
		container.tglSpotEnable.value = light.spotEnabled and 1 or 0

		-- update spotlight direction, exponent, cutoff

		container.valPhiSlide.value 	= light.spotPhi
		container.valPhiSlide.valuechanged_cb()

		container.valThetaSlide.value 	= light.spotTheta
		container.valThetaSlide.valuechanged_cb()

		container.valExpSlide.value 	= light.spotExponent
		container.valExpSlide.valuechanged_cb()

		container.valCutoffSlide.value 	= light.spotCutoff
		container.valCutoffSlide.valuechanged_cb()

		flocker.ui.LightUIAccess( container )

		flocker.ui.refreshDisplay = true
	end

	return lightFrame

end


function flocker.ui.ResetLights()

	-- clear out the old lights
	flocker.ui.lights = {}

	--[[ We use table.insert() below because it gives us back the order
		of objects we want.
		]]

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name 		="Sol"
		, enabled 	= false
		, ambient 	= { 0.2, 0.2, 0.2, 1 }
		, specular	= { 1.0, 1.0, 1.0, 1 }
		, diffuse	= { 0.7, 0.7, 0.7, 1 }
		, position	= { 1, 1, 0, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = {-0.5, -0.5, 0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Mercury"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.0, 0.0, 0.0, 1 }
		, diffuse	= { 0.3, 0.1, 0.1, 1 }
		, position	= { 1, 1, 1, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = {-0.5, -0.5, -0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Venus"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.0, 0.0, 0.0, 1 }
		, diffuse	= { 0.5, 0.5, 0.0, 1 }
		, position	= { 0, 1, 1, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = {0.5, -0.5, -0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Luna"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.0, 0.0, 0.0, 1 }
		, diffuse	= { 0.0, 0.5, 0.5, 1 }
		, position	= { 0, 1, 0, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = { 0.5, -0.5, 0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Mars"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.0, 0.0, 0.0, 1 }
		, diffuse	= { 0.6, 0.0, 0.0, 1 }
		, position	= { 1, 0, 0, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = { -0.5, 0.5, 0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Jupiter"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.0, 0.0, 0.0, 1 }
		, diffuse	= { 0.6, 0.0, 0.6, 1 }
		, position	= { 1, 0, 1, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = { -0.5, 0.5, -0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Saturn"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.0, 0.0, 0.0, 1 }
		, diffuse	= { 0.6, 0.3, 0.0, 1 }
		, position	= { 0, 0, 1, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = { 0.5, 0.5, -0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

	local newLight = flocker.ogl.lighting.CreateLightFromPrototype
	{
		  name		= "Pluto"
		, enabled	= false
		, ambient	= { 0.0, 0.0, 0.0, 1 }
		, specular	= { 0.3, 0.3, 0.3, 1 }
		, diffuse	= { 0.3, 0.3, 0.3, 1 }
		, position	= { 0, 0, 0, 1} -- add the 1 for positional
		, spotEnabled = true
		, spotDirection = { 0.5, 0.5, 0.5, 1}
		, spotCutoff	= 30
		, spotExponent	= 0
	}
	table.insert(flocker.ui.lights, newLight)

end

flocker.ui.ResetLights()
-- create controller and point it at the first light

flocker.ui.lightController = flocker.ui.CreateLightController( flocker.ui.lights[1] )


function flocker.ui.GenerateLightTabber()

	-- static configuration
	local tabConfig =
	{
		  tabtype = "TOP"
		, multiline="YES"
		, tabchange_cb = function(self, new_tab, old_tab)

				-- must convert from string to number for correct indexing
				local index = tonumber(new_tab.lightId)
				local newLight = flocker.ui.lights[ index ]
				flocker.ui.lightController.UpdateLightControls( newLight )
				flocker.ui.refreshDisplay = true
			end
	}

	-- dynamic configuration
	for lightId, light in pairs(flocker.ui.lights) do

		-- create a label for each light
		local iupLabel = iup.label
		{	  alignment = "ACENTER"
			, expand	= "YES"
			, lightId 	= lightId
			, tabtitle	= light.name
			, title 	= string.format("Configuring OpenGL Light %d", light.index )
		}

		-- insert that into the tab configuration table
		table.insert(tabConfig, iupLabel)
	end

	-- creae the tabber object
	local tabber = iup.tabs( tabConfig )

	return tabber

end

flocker.ui.lightControllerTabs = flocker.ui.GenerateLightTabber()

flocker.ui.lightingConfig = iup.frame
	{
		  tabtitle="Lighting", expand="YES"
		, iup.vbox
		{	  expand="YES"
			, iup.hbox { flocker.ui.btnGlobalAmbient, flocker.ui.helpGlobalAmbient, flocker.ui.txtGlobalAmbient }
			, flocker.ui.lightControllerTabs
			, flocker.ui.lightController
		}
	}

function flocker.ui.DeleteLightSettings()

	flocker.db.DeleteBlob( "lightsettings", "Delete lighting configurations." )

end

function flocker.ui.LoadLightSettings( lightSettingNameMatcher )

	local settings = flocker.db.LoadBlob("lightsettings","Choose a lighting configuration.", lightSettingNameMatcher)

	if not settings then return end

	local newSettings = decodeLSONTable( settings )
	assert(newSettings, "Failed to reconstruct the settings.")

	local lighting = flocker.ogl.lighting

	lighting:ClearLights()
	lighting:SetGAmbient( newSettings.globalAmbient )

	for identifier, lightProto in pairs(newSettings.lights) do

		flocker.ui.lights[identifier] = lighting.CreateLightFromPrototype( lightProto )
		newSettings.lights[identifier]=nil	-- allow the object wrapper to be collected
	end

	-- tell the tab object to go back to the first tab
	flocker.ui.lightControllerTabs.valuepos = 0
	--update the light controls
	flocker.ui.lightController.UpdateLightControls( flocker.ui.lights[ 1 ] )
	flocker.ui.refreshDisplay = true

end

function flocker.ui.SetDefaultLightSettings()

	local lighting = flocker.ogl.lighting
	lighting:ClearLights()
	lighting:SetGAmbient()

	flocker.ui.ResetLights()

	-- tell the tab object to go back to the first tab
	flocker.ui.lightControllerTabs.valuepos = 0
	flocker.ui.lightController.UpdateLightControls( flocker.ui.lights[ 1 ] )
	--update the light controls
	flocker.ui.refreshDisplay = true

end

function flocker.ui.SaveLightSettings()
	--[[
		(1) grab the global settings and the four lights and stuff into an object
		(2) serlize that to LSON
		(3) call flocker.db.SaveBlob
		]]

	local wrapper =
	{
		  globalAmbient = flocker.ogl.lighting.globalAmbient
		, lights = {}
	}
	for identifier, light in pairs(flocker.ui.lights) do
		wrapper.lights[identifier] = light
	end

	local lsonText = encodeLSONTable(wrapper)

	flocker.db.SaveBlob("lightsettings","Save a lighting configuration.", lsonText)
end



flocker.ui.birdConfig =
	iup.frame
	{
		title=nil --"Configuration"
		, tabtitle = "Birds"
		, expand="YES"
		, iup.vbox
		{	 alignment="ACENTER"
			, iup.label{ separator }
			, iup.hbox { flocker.ui.valRepelSlide, 			iup.frame{ flocker.ui.lblRepel 		} }
			, iup.hbox { flocker.ui.valDetectSlide, 		iup.frame{ flocker.ui.lblDetect 	} }
			, iup.hbox { flocker.ui.valCaffeineSlide, 		iup.frame{ flocker.ui.lblCaffeine 	} }
			, iup.hbox { flocker.ui.valAgilitySlide, 		iup.frame{ flocker.ui.lblAgility 	} }
			, iup.hbox { flocker.ui.valAntiSocialitySlide,	iup.frame{ flocker.ui.lblAntiSociality } }
			, iup.hbox { flocker.ui.valDeltaPXSlide, 		iup.frame{ flocker.ui.lblDeltaPX	} }
			, iup.hbox { flocker.ui.valDeltaTSlide, 		iup.frame{ flocker.ui.lblDeltaT		} }

			, iup.label{ separator }
			, iup.hbox {
				  expand="YES"
				, iup.fill{}
				, flocker.ui.tglPartisanShip
				, iup.fill{}
				, flocker.ui.tglAggressive
				, iup.fill{}
			}
		}
	}

flocker.ui.experimental = iup.frame
	{
		title = "EXPERIMENTAL"
		, expand="YES"
		, iup.vbox
		{
			alignment="ACENTER"
			, iup.label{separator="HORIZONTAL"}
			, flocker.ui.tglEnableFog
			, iup.hbox
			{
				  flocker.ui.valFogIntensitySlide
				, iup.frame{flocker.ui.lblFogIntensity}
			}
		}
	}


flocker.ui.birdDisplay =
	iup.frame
	{
		title= nil --"Display"
		, tabtitle = "Display"
		, expand="YES"
		, iup.vbox
		{
			  alignment="ACENTER"
			, iup.label{ separator }
			, iup.hbox {
				  expand="YES"
				, iup.fill{}
				, flocker.ui.tglShowRepel
				, iup.fill{}
				, flocker.ui.tglShowDetect
				, iup.fill{}
			}
			, iup.label{ separator }
			, iup.hbox
			{	   iup.label{title="Background:", alignment="ACENTER:ACENTER" }
				, flocker.ui.lstBgColor
			}
			, iup.label{ separator }
			, flocker.ui.tglReflectBirds
			, iup.label{ separator }
			, flocker.ui.tglThreeDeeFlocking
			, iup.hbox
			{
				flocker.ui.valGapSlide
				, iup.frame{ flocker.ui.lblGap }
			}
			, iup.hbox
			{
				flocker.ui.valFOVSlide
				, iup.frame{ flocker.ui.lblFOV }
			}
			, iup.hbox
			{
				  flocker.ui.valZDepthSlide
				, iup.frame{flocker.ui.lblZDepth}
			}
			, iup.fill{}
		--	, flocker.ui.experimental
		}
	}

flocker.ui.birdStatistics =
	iup.text
	{
		tabtitle = "Statistics"
		, font = "Courier New 14"
		, multiline = "YES"
		, readonly  = "YES"
		, expand	= "YES"
		, border	= "YES"
	}

-- namespace for display world stuff
flocker.ui.world ={}

function flocker.ui.world.focus_cb(self, focus)
	flocker.cursorFocus = 0 ~= focus
end

function flocker.ui.world.wheel_cb(self, delta, x, y, status)

	local camera = flocker.ogl.camera

	camera.distance = camera.distance - delta*10

	if camera.distance < 1 then camera.distance = 1 end

	flocker.ui.refreshDisplay = true
end


flocker.ui.buttonDown 	= { button=0, pos=vec3.new(x,y,0), status = 0 }
flocker.ui.world.lastCursor = vec3.new(1, 1, 0)

function flocker.ui.world.motion_cb(self, x,y)

	-- compensate for coordinate system conversion
	x = flocker.toroidX - x
	y = flocker.toroidY - y

	-- save new cursor position
	vec3.set( flocker.cursorPos, x, y, 0)

	if x < 0 or y < 0 or x > flocker.toroidX or y > flocker.toroidY then return end

	if flocker.threeDeeFlocking then

		local camera = flocker.ogl.camera
		local avatar = flocker.ogl.avatar
		if flocker.trackCamera then

			local delta = vec3.sub( flocker.cursorPos, 	flocker.ui.world.lastCursor )

			vec3.copyTo( flocker.ui.world.lastCursor, flocker.cursorPos)

			local dPitch, dYaw =  -delta[2]*140/360,  delta[1]*140/360

			if iup.BUTTON3 == flocker.ui.buttonDown.button then

				-- give the avatar the same speed as the birds
				avatar:SetSpeed( flocker.ui.advantage * flocker.deltaPX )
				-- pitch and yaw are incrementally adjusted
				avatar:AccumulatePitchAndYaw( dPitch, dYaw )

				camera:SetPitchAndYaw(avatar.pitch, avatar.yaw)

			elseif iup.BUTTON1 == flocker.ui.buttonDown.button then

				-- rotate POV around avatar
				camera:AccumulatePitchAndYaw( dPitch, dYaw )
			end


			flocker.ui.refreshDisplay = true
		elseif not flocker.ui.autoRun then

			avatar:SetSpeed( 0 )

		end

	else -- 2D display for bird position manipulation

		if flocker.trackCursor then
			-- if the cursor moved
			if not vec3.equal(		flocker.cursorPos,	flocker.ui.world.lastCursor ) then
				-- try averaging in leaky integrator fashion so smooth out direction changes
				local newDir = vec3.sub( flocker.cursorPos, 	flocker.ui.world.lastCursor )
				vec3.normalizeIt( newDir )
				vec3.scaleIt( newDir, 0.2 )

				vec3.addTo(			flocker.cursorDir, 	newDir )
				vec3.normalizeIt(	flocker.cursorDir	)
			end

			-- save the old one for calculating direction
			vec3.copyTo( flocker.ui.world.lastCursor, flocker.cursorPos)

			-- need to do this if simulation is not running
			if not flocker.ui.runningSimulation then
				flocker.MoveCaughtBirds( )
			end
			flocker.ui.refreshDisplay = true
		end
	end
end


function flocker.ui.world.keypress_cb(self, c, pressed )

	if flocker.threeDeeFlocking then
		local camera = flocker.ogl.camera
		local avatar = flocker.ogl.avatar

		if 1 == pressed then

			if c == iup.K_UP then

				flocker.ui.autoRun = true
				avatar:SetSpeed( (flocker.deltaPX>3) and flocker.ui.advantage * flocker.deltaPx or 5 )

			elseif c == iup.K_DOWN then

				flocker.ui.autoRun = false
				avatar:SetSpeed( 0 )

			elseif c == iup.K_LEFT then

				avatar:AccumulatePitchAndYaw( 0, 2 )
				camera:SetPitchAndYaw(avatar.pitch, avatar.yaw)

			elseif c == iup.K_RIGHT then

				avatar:AccumulatePitchAndYaw( 0, -2 )
				camera:SetPitchAndYaw(avatar.pitch, avatar.yaw)

			end

		end
	end
end

function flocker.ui.world.button_cb(self, button, pressed, x, y, status)

	-- compensate for coordinate system conversion
	x = flocker.toroidX - x
	y = flocker.toroidY - y
	vec3.set( flocker.cursorPos, x, y, 0)

	local camera = flocker.ogl.camera
	local avatar = flocker.ogl.avatar

	if 1 == pressed then
		flocker.ui.buttonDown 	= { button=button, pos=vec3.new(x,y,0), status = status }

	else
		flocker.ui.buttonUp 	= { button=button, pos=vec3.new(x,y,0), status = status }
	end

	if flocker.threeDeeFlocking then
		-- we may be adjusting the viewpoint
		if 1 == pressed then
			flocker.trackCamera 	= true
			flocker.ui.world.iupglcanvas.cursor = "CROSS"
			vec3.set( flocker.ui.world.lastCursor, x, y, 0)
			-- do like WoW : set the new travel direction to the view direcion
			if iup.BUTTON3 == flocker.ui.buttonDown.button then

				avatar:SetPitchAndYaw(camera.pitch, camera.yaw)

			end
		else
			flocker.trackCamera 	= false
			flocker.ui.world.iupglcanvas.cursor = "ARROW"
		end

	else

		if 1 == pressed then
			flocker.trackCursor 	= true
			flocker.CatchBirds( 	flocker.ui.buttonDown.pos )
		else
			flocker.trackCursor 	= false
			if not vec3.equal(		flocker.ui.buttonUp.pos,	flocker.cursorPos) then
				vec3.copyTo( 		flocker.cursorDir, 			flocker.ui.buttonUp.pos  )
				vec3.subFrom( 		flocker.cursorDir, 			flocker.cursorPos)
				vec3.normalizeIt( 	flocker.cursorDir )
			end
			flocker.ReleaseBirds()
		end

	end

	flocker.ui.refreshDisplay = true
end

function flocker.ui.world.resize_cb(self, width, height )
	-- fix inability to restrict resizing of the UI to minsize
	if height < 400 then height = 400 end
	if width < 400 then width = 400 end
	flocker.toroidX = tonumber(width)
	flocker.toroidY = tonumber(height)

	-- force them to be inside the volume
	flocker.WrapBirds()

	-- redisplay
	flocker.ui.refreshDisplay = true
end



	--[[ ===================== OPENGL DISPLAY ===================== ]]
flocker.ui.world.iupglcanvas = iup.glcanvas
	{
		  cursor="ARROW"
		, tabTitle = "OpenGL Bird World"

		, buffer="DOUBLE"
		, rastersize = "500x500" -- is updated by resize_cb on display

		, wheel_cb	= flocker.ui.world.wheel_cb
		, button_cb	= flocker.ui.world.button_cb
		, keypress_cb = flocker.ui.world.keypress_cb
		, focus_cb	= flocker.ui.world.focus_cb
		, resize_cb	= flocker.ui.world.resize_cb
		, motion_cb	= flocker.ui.world.motion_cb

		, Iterate = flocker.FlockBirdsOhNK	-- this does the work
		, RefreshDisplay = function() flocker.ogl.RefreshDisplay(flocker.ui.world.iupglcanvas) end

		, map_cb = function() --[[ placeholder ]] end
	}


	--[[ ==================== TEST INFRASTRUCTURE ================= ]]
flocker.ui.world.iupogltest = iup.glcanvas
	{
		  cursor="ARROW"
		, tabTitle = "OpenGL Testing"

		, buffer="DOUBLE"
		, rastersize = "500x500" -- is updated by resize_cb on display

		, wheel_cb	= flocker.ui.world.wheel_cb
		, button_cb	= flocker.ui.world.button_cb
		, focus_cb	= flocker.ui.world.focus_cb
		, resize_cb	= flocker.ui.world.resize_cb
		, motion_cb	= flocker.ui.world.motion_cb

		, angle = 0
		, delta = math.pi/180
		, p1 = 0
		, p2 = math.pi/2
		, counter = 1

		, Iterate = function() local self = flocker.ui.world.iupogltest; self.angle = self.angle + self.delta end
		, RefreshDisplay = function() flocker.ogl.TestFixture(flocker.ui.world.iupogltest) end
	}




flocker.ui.startConfig =
	{
		  startSize = "636x455" --"HALFxHALF"
		, startX = iup.CENTER
		, startY = iup.CENTER
	}

local config = flocker.db.GrabUIConfig()
if config then
	flocker.ui.startConfig.startSize= config.size or flocker.ui.startConfig.startSize
	flocker.ui.startConfig.startX 	= config.posx or flocker.ui.startConfig.startX
	flocker.ui.startConfig.startY 	= config.posy or flocker.ui.startConfig.startY
end



function flocker.ui.syncUI()

	flocker.ui.valRepelSlide.value			= flocker.repulseR
	flocker.ui.valRepelSlide:valuechanged_cb()

	flocker.ui.valDetectSlide.value			= flocker.detectR
	flocker.ui.valDetectSlide:valuechanged_cb()

	flocker.ui.valCaffeineSlide.value		= flocker.caffeine
	flocker.ui.valCaffeineSlide:valuechanged_cb()

	flocker.ui.valAgilitySlide.value		= flocker.agility
	flocker.ui.valAgilitySlide:valuechanged_cb()

	flocker.ui.valAntiSocialitySlide.value	= flocker.antisociality
	flocker.ui.valAntiSocialitySlide:valuechanged_cb()

	flocker.ui.valDeltaPXSlide.value		= flocker.deltaPX
	flocker.ui.valDeltaPXSlide:valuechanged_cb()

	flocker.ui.valDeltaTSlide.value			= flocker.deltaT
	flocker.ui.valDeltaTSlide:valuechanged_cb()

	flocker.ui.tglPartisanShip.value 		= flocker.partisan and 1 or 0 -- funky LUA ternary operator
	flocker.ui.tglAggressive.value 			= flocker.aggressive and 1 or 0 -- funky LUA ternary operator
	--[[
		The IUP control expects 1/0 and not true/false. LUA considers nil and false
		to be false, and I need to convert to 1/0,
		Boolean data types are always a pain 'cuz everyone wants to be different. :)
		]]
	flocker.ui.tglShowRepel.value 	= flocker.showRepel  and 1 or 0 -- funky LUA ternary operator
	flocker.ui.tglShowDetect.value 	= flocker.showDetect and 1 or 0 -- funky LUA ternary operator
	flocker.ui.lstBgColor.value 	= flocker.bgColor

	flocker.ui.tglThreeDeeFlocking.value	= flocker.threeDeeFlocking 	and 1 or 0
	flocker.ui.tglEnableFog.value			= flocker.ogl.enableFog 	and 1 or 0
	flocker.ui.tglReflectBirds.value		= flocker.reflectBirds 		and 1 or 0

	flocker.ui.valFogIntensitySlide.value = flocker.ogl.fogIntensity
	flocker.ui.valFogIntensitySlide:valuechanged_cb()

	flocker.ui.valZDepthSlide.value = flocker.toroidZ
	flocker.ui.valZDepthSlide:valuechanged_cb()


	flocker.ui.lblBirdCount.title = string.format("%d birds", flocker.birdCount)

end

flocker.test = {}
function flocker.test.ExplodingBirds( birdCount, threeD, reflected, repulse, detect )

	-- setup for testing
	flocker.threeDeeFlocking	= threeD
	flocker.reflectBirds		= reflected
	flocker.repulseR			= repulse or 20
	flocker.detectR				= detect  or 60
	local center 				= { flocker.toroidX/2, flocker.toroidY/2, flocker.toroidZ/2 }

	-- spawn the chicks
	flocker.ClearBirds()
	flocker.ogl.avatar:SetPosition(flocker.toroidX/2, flocker.toroidY/2, 1)

	for k = 1,birdCount do
		flocker.createNewBird( flocker.birdTypes[1], center, nil, 10000, 1 )
	end
end


function flocker.test.ParallelBirds( birdCount, threeD, reflected, repulse, detect )

	-- setup for testing
	flocker.threeDeeFlocking	= threeD
	flocker.reflectBirds		= reflected
	flocker.repulseR			= repulse or 20
	flocker.detectR				= detect  or 60
	local direction 			= { 1,0,0 }

	flocker.caffeine			= 0
	flocker.agility				=5

	-- spawn the chicks
	flocker.ClearBirds()
	flocker.ogl.avatar:SetPosition(flocker.toroidX/2, flocker.toroidY/2, 1)

	for k = 1,birdCount do
		flocker.createNewBird( flocker.birdTypes[1], nill, direction, 10000, 1 )
	end
end

function flocker.test.QuickTest( birdGeneratingFunction, ... )
	birdGeneratingFunction( unpack( ... ) )
	flocker.ui.syncUI()
	flocker.ui.refreshDisplay = true
	return iup.DEFAULT
end


-- Creates menu with items, and shortcut keys
flocker.ui.menu = iup.menu
	{
		iup.submenu
		{
			title = "&File", minsize="20x"
			, iup.menu
			{
				 iup.separator{ }
				, iup.item
				{
					  title = "&Compact database"
					, action = function() flocker.db.VacuumDatabase(); return iup.DEFAULT end
				}

				, iup.separator{ }
				, iup.item
				{
					  title = "&Exit", key = "K_x"
					, action = function() return iup.CLOSE end
				}
			}
		}
		, iup.submenu
		{
			title = "&Birds"
			, iup.menu
			{
				 iup.submenu
				{
					title="&Canned Tests"
					, iup.menu
					{
						  iup.item
						{
							  title = "FPS test: Exploding 3D Birds x 300"
							, action = function() flocker.test.QuickTest( flocker.test.ExplodingBirds, {300, true, false, 20, 60} ) end
						}
						,iup.item
						{
							  title = "FPS test: Parallel 3D Birds x 300"
							, action = function() flocker.test.QuickTest( flocker.test.ParallelBirds, { 300, true, false, 20, 60 } ) end
						}
						,iup.item
						{
							  title = "Exploding 3D Birds x 100"
							, action = function() flocker.test.QuickTest( flocker.test.ExplodingBirds, {100, true, false, 30, 40 } ) end
						}
						,iup.item
						{
							  title = "Exploding 3D Birds x 300"
							, action = function() flocker.test.QuickTest( flocker.test.ExplodingBirds, {300, true, false, 30, 40 } ) end
						}
						,iup.item
						{
							  title = "Exploding 3D Birds x 100, reflected"
							, action = function() flocker.test.QuickTest( flocker.test.ExplodingBirds, { 100, true, true, 30, 40 } ) end
						}
						,iup.item
						{
							  title = "Exploding 3D Birds x 300, reflected"
							, action = function() flocker.test.QuickTest( flocker.test.ExplodingBirds, { 300, true, true, 30, 40 } ) end
						}
						, iup.item
						{
							  title = "Parallel 3D Birds x 100"
							, action = function() flocker.test.QuickTest( flocker.test.ParallelBirds, { 100, true, false, 30, 40 } ) end
						}
						, iup.item
						{
							  title = "Parallel 3D Birds x 300"
							, action = function() flocker.test.QuickTest( flocker.test.ParallelBirds, { 300, true, false, 30, 40 } ) end
						}
						,iup.item
						{
							  title = "Parallel 3D Birds x 100, reflected"
							, action = function() flocker.test.QuickTest( flocker.test.ParallelBirds, { 100, true, true, 20, 40 } ) end
						}
						,iup.item
						{
							  title = "Parallel 3D Birds x 300, reflected"
							, action = function() flocker.test.QuickTest( flocker.test.ParallelBirds, { 300, true, true, 20, 40 } ) end
						}
					}
				}
				, iup.separator{}
				, iup.item
				{
					  title = "&Load Snapshot"
					, action = function()
						flocker.db.LoadSnapshot()
						flocker.ui.syncUI()
						flocker.ui.refreshDisplay = true
						return iup.DEFAULT
					end
				}
				, iup.separator{}
				, iup.item
				{
					  title = "&Save Snapshot As..."
					, action = function() flocker.db.SaveSnapshot(); return iup.DEFAULT end
				}
				, iup.separator{}
				, iup.item
				{
					  title = "&Delete Snapshot"
					, action = function() flocker.db.DeleteSnapshot(); return iup.DEFAULT end
				}
			}
		}
		, iup.submenu
		{
			title = "&Lights"
			, iup.menu
			{

				 iup.item
				{
					  title = "&Load Light Settings"
					, action = function() flocker.ui.LoadLightSettings(); return iup.DEFAULT end
				}
				, iup.separator{}
				, iup.item
				{
					  title = "&Save Light Settings As..."
					, action = function() flocker.ui.SaveLightSettings(); return iup.DEFAULT end
				}
				, iup.separator{}
				, iup.item
				{
					  title = "&Delete Light Settings"
					, action = function() flocker.ui.DeleteLightSettings(); return iup.DEFAULT end
				}
				, iup.separator{}
				, iup.item
				{
					  title = "&Reset Light Defaults"
					, action = function() flocker.ui.SetDefaultLightSettings(); return iup.DEFAULT end
				}


			}
		}
	}

flocker.ui.statsTimer = iup.timer { time = 1000, run="yes" }
flocker.ui.statsTimer.lastCounter = 0
flocker.ui.statsTimer.action_cb = function()

	-- calculate FPS given that we run this function once per second
	local frames = flocker.iterations - flocker.ui.statsTimer.lastCounter
	flocker.ui.statsTimer.lastCounter = flocker.iterations

	--[[
		Gather statistics, take pains to preserve the desired
		order, but also add in any unexpected ages, types, or states
		]]

	local aNames = { "young", "adult", "mature", "declining" }
	local aCounters = { young = 0, adult = 0, mature = 0, declining = 0 }

	local sNames = { "idle", "flee", "fight", "pregnant", "dead" }
	local sCounters = { idle = 0, flee = 0, fight = 0, pregnant = 0, dead = 0 }

	local tNames = { "type1", "type2", "hybrid" }
	local tCounters = { type1 = 0, type2 = 0, hybrid = 0 }

	for __, bird in pairs( flocker.birds ) do
		local aname   = bird.age
		if not aCounters[aname]   then
			aCounters[aname]=1
			table.insert( aNames, aname)
		else
			aCounters[aname] = aCounters[aname]   + 1
		end

		local sname = bird.state.name
		if not sCounters[sname] then
			sCounters[sname] = 1
			table.insert( sNames, sname)
		else
			sCounters[sname] = sCounters[sname] + 1
		end

		local tname = bird.typeName
		if not tCounters[tname] then
			tCounters[tname] = 1
			table.insert( tNames, tname)
		else
			tCounters[tname] = tCounters[tname] + 1
		end
	end

	local displayString = string.format("Frames/sec    %d\n", frames)

	displayString = displayString .. "Types\n"
	for __, name in ipairs(tNames) do
		displayString = displayString .. string.format("  %-12.12s%d\n", name, tCounters[name])
	end

	displayString = displayString .. "Ages\n"
	for __, name in ipairs(aNames) do
		displayString = displayString .. string.format("  %-12.12s%d\n", name, aCounters[name])
	end

	displayString = displayString .. "States\n"
	for __, name in ipairs(sNames) do
		displayString = displayString .. string.format("  %-12.12s%d\n", name, sCounters[name])
	end

	flocker.ui.birdStatistics.value = displayString
end


flocker.ui.lblSimulationTime = iup.label
	{
		title="0 s (0 steps)"
		, minsize="120x"
		, padding="3x3"
		, alignment="ACENTER:ACENTER"
	}


flocker.ui.btnStep = iup.button
	{
		title = "Step"
		, tip = "Take one step of the simulation"
		, expand="HORIZONTAL"
		, action = function()
			if not flocker.ui.runningSimulation then
				flocker.simulator.Iterate()
				-- because birds can change state and graphics
				flocker.ui.refreshDisplay = true
				flocker.ui.RefreshDisplay( )
			end
			return iup.DEFAULT
		end

	}


flocker.ui.btnStart = iup.button
	{
		title = "Start"
		, tip = "Start (and stop, or pause) the simulation"
		, expand="HORIZONTAL"
		, action = function()
			if flocker.ui.runningSimulation then
				flocker.ui.runningSimulation	= false
				flocker.ui.btnStart.title 		= "Start"

				--enable/disable buttons]]
				flocker.ui.btnStep.active 		= "YES"
			else
				flocker.ui.runningSimulation	= true
				flocker.ui.btnStart.title 		= "Stop"

				--enable/disable buttons
				flocker.ui.btnStep.active 		= "NO"
			end
			return iup.DEFAULT
		end

	}


--the main dialog window
flocker.ui.dialog = iup.dialog
	{
		title="Flocker 2.2.1 (c) 2010 Eric Fredericksen - www.pttpsystems.com"
		, margin="1x1", padding="1x1", gap=4
		, shrink="YES"
		, menu = flocker.ui.menu
		--, fontsize="12"
		--, opacity=200
		, size=flocker.ui.startConfig.startSize
		, minsize="HALFxHALF"
		, show_cb = function(self)
				if not self.init then
					self.init = true
					flocker.ClearBirds()
					flocker.ogl.avatar:SetPosition(flocker.toroidX/2, flocker.toroidY/2, 1)
					flocker.db.LoadSnapshot( "Startup%" )
					flocker.ui.LoadLightSettings( "Startup%" )
					flocker.ui.syncUI()
					flocker.ui.refreshDisplay = true
				end
			end
		, iup.hbox
		{
			iup.vbox
			{
				expand="NO"
				, iup.frame
				{
					title="Simulation", expand="YES"
					, iup.vbox
					{
						iup.hbox
						{
							  flocker.ui.btnStart
							, flocker.ui.btnStep
							, iup.frame{ flocker.ui.lblSimulationTime }
						}
						, iup.hbox
						{
							  flocker.ui.btnAddBird
							, flocker.ui.btnClearBirds
							, iup.frame{ flocker.ui.lblBirdCount }
						}
					}
				}
				, iup.tabs
				{
					  flocker.ui.birdConfig
					, flocker.ui.birdDisplay
					, flocker.ui.lightingConfig
					, flocker.ui.birdStatistics
				}
			}
			, iup.tabs
			{
				  bgcolor="0 0 0"
				, flocker.ui.world.iupglcanvas
				, flocker.ui.world.iupogltest

				, tabchange_cb = function(self, new_tab, old_tab)
						flocker.simulator = new_tab
						flocker.ui.refreshDisplay = true
					end
			}
		}
	}

-- set the initial simulator
flocker.simulator = flocker.ui.world.iupglcanvas

-- so that we get different and new RNG seed
local datetime = os.date( "*t")
local seed = datetime.sec * datetime.hour * datetime.day * datetime.month * datetime.year
math.randomseed( seed )

function flocker.ui.dialog.resize_cb(self, width, height )
	-- needed this for CenterIt()
	flocker.ui.dialog.sizeX = tonumber(width)
	flocker.ui.dialog.sizeY = tonumber(height)
end

function flocker.ui.dialog.CenterIt( boxW, boxH )
	local dlg = flocker.ui.dialog
	local x = math.floor( 0.5 + dlg.X + dlg.sizeX/2 - boxW/2 )
	local y = math.floor( 0.5 + dlg.Y + dlg.sizeY/2 - boxH/2 )
	return x,y
end

-- instrument the close action to save UI configuration
function flocker.ui.dialog:close_cb()
	flocker.db.SaveUIConfig( self.SIZE, self.X, self.Y )
end

-- show the dialog to the user in the last known position
flocker.ui.dialog:showxy( flocker.ui.startConfig.startX, flocker.ui.startConfig.startY )

-- go into the main loop
iup:MainLoop()

